<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';

requireLogin();
header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit();
}

try {
    // Validate required fields
    $required_fields = ['customer_id', 'issue_date', 'due_date'];
    foreach ($required_fields as $field) {
        if (!isset($_POST[$field]) || empty($_POST[$field])) {
            echo json_encode(['success' => false, 'message' => "Missing required field: $field"]);
            exit();
        }
    }

    // Validate items
    if (!isset($_POST['product_id']) || !is_array($_POST['product_id']) || empty($_POST['product_id'])) {
        echo json_encode(['success' => false, 'message' => 'At least one product is required']);
        exit();
    }

    $product_ids = $_POST['product_id'];
    $quantities = $_POST['quantity'];
    $unit_prices = $_POST['unit_price'];
    $total_prices = $_POST['total_price'];

    // Validate arrays have same length
    if (count($product_ids) !== count($quantities) || 
        count($quantities) !== count($unit_prices) || 
        count($unit_prices) !== count($total_prices)) {
        echo json_encode(['success' => false, 'message' => 'Invalid item data']);
        exit();
    }

    // Start transaction
    $pdo->beginTransaction();

    // Generate invoice number
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM invoices WHERE DATE(created_at) = CURDATE()");
    $stmt->execute();
    $today_count = $stmt->fetchColumn();
    $invoice_number = 'INV-' . date('Ymd') . '-' . str_pad($today_count + 1, 3, '0', STR_PAD_LEFT);

    // Calculate totals
    $subtotal = 0;
    $items_data = [];
    
    for ($i = 0; $i < count($product_ids); $i++) {
        if (empty($product_ids[$i])) continue; // Skip empty items
        
        $quantity = floatval($quantities[$i]);
        $unit_price = floatval($unit_prices[$i]);
        $total_price = floatval($total_prices[$i]);
        
        if ($quantity <= 0 || $unit_price <= 0) {
            throw new Exception('Invalid quantity or price for item ' . ($i + 1));
        }
        
        $subtotal += $total_price;
        $items_data[] = [
            'product_id' => $product_ids[$i],
            'quantity' => $quantity,
            'unit_price' => $unit_price,
            'total_price' => $total_price
        ];
    }

    if (empty($items_data)) {
        throw new Exception('At least one valid product is required');
    }

    $tax_rate = 0.00; // 0% tax
    $tax_amount = $subtotal * $tax_rate;
    $total_amount = $subtotal + $tax_amount;

    // Create invoice
    $stmt = $pdo->prepare("
        INSERT INTO invoices (
            invoice_number, customer_id, issue_date, due_date, 
            subtotal, tax_amount, total_amount, status, 
            notes, created_by, created_at
        ) VALUES (?, ?, ?, ?, ?, ?, ?, 'pending', ?, ?, NOW())
    ");

    $notes = $_POST['notes'] ?? '';
    $stmt->execute([
        $invoice_number,
        $_POST['customer_id'],
        $_POST['issue_date'],
        $_POST['due_date'],
        $subtotal,
        $tax_amount,
        $total_amount,
        $notes,
        $_SESSION['user_id']
    ]);

    $invoice_id = $pdo->lastInsertId();

    // Insert invoice items
    $stmt = $pdo->prepare("
        INSERT INTO invoice_items (
            invoice_id, product_id, quantity, unit_price, total_price
        ) VALUES (?, ?, ?, ?, ?)
    ");

    foreach ($items_data as $item) {
        $stmt->execute([
            $invoice_id,
            $item['product_id'],
            $item['quantity'],
            $item['unit_price'],
            $item['total_price']
        ]);
    }

    // Update quotation status if invoice was created from quotation
    if (!empty($_POST['quotation_id'])) {
        $stmt = $pdo->prepare("
            UPDATE quotations 
            SET status = 'converted', 
                updated_at = NOW() 
            WHERE id = ?
        ");
        $stmt->execute([$_POST['quotation_id']]);
    }

    // Commit transaction
    $pdo->commit();

    echo json_encode([
        'success' => true,
        'message' => "Invoice $invoice_number created successfully!",
        'invoice_number' => $invoice_number,
        'invoice_id' => $invoice_id
    ]);

} catch (Exception $e) {
    // Rollback transaction on error
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    echo json_encode([
        'success' => false,
        'message' => 'Error creating invoice: ' . $e->getMessage()
    ]);
} 